<?php
/**
 * Functions
 *
 * @since 1.0.0
 *
 * @package Favorites
 * @category Functions
 * @author Astoundify
 */

/**
 * Favorite Query.
 *
 * @since 1.2.0
 *
 * @param array $args Query Args.
 * @return array Array of favorite object.
 */
function astoundify_favorites_get_favorites( $args = array() ) {
	$_favorites = array();
	$favorites = new \Astoundify\Favorites\Favorite_Query( $args );
	foreach ( $favorites->favorites as $favorite ) {
		$_favorites[ $favorite->get_id() ] = $favorite;
	}
	return $_favorites;
}

/**
 * Supported Post Types
 *
 * @since 1.0.0
 */
function astoundify_favorites_post_types() {
	return apply_filters( 'astoundify_favorites_post_types', array(
		'post',
	) );
}

/**
 * Get Options
 *
 * @since 1.0.0
 *
 * @param string $option Required.
 * @param string $default Optional.
 * @param string $option_name Optional default to `astoundify_favorites`.
 * @return mixed
 */
function astoundify_favorites_get_option( $option, $default = '', $option_name = 'astoundify_favorites' ) {
	if ( ! $option ) {
		return false;
	}

	$get_option = get_option( $option_name );

	if ( ! is_array( $get_option ) ) {
		return $default;
	}

	if ( isset( $get_option[ $option ] ) ) {
		return $get_option[ $option ];
	} else {
		return $default;
	}
}


/**
 * Get Template File : For PHP File
 *
 * @since 1.0.0
 * @link https://developer.wordpress.org/reference/functions/locate_template/
 * @link https://developer.wordpress.org/reference/functions/load_template/
 *
 * @param string $template_name The filename of the template to load.
 * @param array  $data Variable to use in template.
 * @return void
 */
function astoundify_favorites_get_template( $template_name, $data = array() ) {
	// Get theme template file.
	$template = locate_template( "astoundify-favorites/{$template_name}.php", false, false );

	// Theme template file not found, use default plugin template.
	if ( ! $template ) {
		$template = trailingslashit( ASTOUNDIFY_FAVORITES_PATH . 'resources/templates' ) . $template_name . '.php';
	}

	// Check file exists.
	if ( file_exists( $template ) ) {

		// Filter data.
		$data = apply_filters( 'astoundify_favorite_template_data', $data, $template_name );

		// Extract vars for easy use.
		if ( $data && is_array( $data ) ) {
			extract( $data );
		}

		// Load template.
		include( $template );
	}
}

/**
 * See if a post is favorited by a user
 *
 * @since 1.0.0
 * @link https://developer.wordpress.org/reference/functions/get_posts/
 *
 * @param  int $target_id Target Post ID.
 * @param  int $user_id Favorite Author User ID.
 * @return bool|int False if not favorited, Fav ID if already favorited
 */
function astoundify_favorites_is_favorited( $target_id, $user_id ) {
	// User need to logged in.
	if ( ! $target_id || ! $user_id ) {
		return false;
	}

	$args = array(
		'target_id'      => $target_id,
		'user_id'        => $user_id,
		'item_per_page'  => 1,
		'fields'         => 'ids',
	);

	$favorite_query = new \Astoundify\Favorites\Favorite_Query( $args );

	return $favorite_query->favorites ? $favorite_query->favorites[0] : false;
}

/**
 * Favorite Count
 *
 * @since 1.0.0
 *
 * @param int $target_id ID of the WordPress object to query.
 */
function astoundify_favorites_count( $target_id ) {
	$count = get_post_meta( $target_id, '_astoundify_favorites_count', true );

	// No count, recalculate.
	if ( ! $count ) {
		$args = array(
			'target_id'      => $target_id,
			'fields'         => 'ids',
			'item_per_page'  => -1,
			'user_id'        => false,
		);

		$favorite_query = new \Astoundify\Favorites\Favorite_Query( $args );
		$count = $favorite_query->total_items;

		// Update meta.
		update_post_meta( $target_id, '_astoundify_favorites_count', absint( $count ) );
	}

	return absint( $count );
}

/**
 * Favorite Count HTML
 *
 * @since 1.0.0
 *
 * @param int $target_id ID of the WordPress object to query.
 */
function astoundify_favorites_count_html( $target_id ) {
	return '<span class="astoundify-favorites-count">' . astoundify_favorites_count( $target_id ) . '</span>';
}

/**
 * User Favorites Count
 *
 * @since 1.0.0
 *
 * @param int $user_id ID of user.
 * @return int
 */
function astoundify_favorites_user_favorites_count( $user_id = 0 ) {

	// Use current user if not set.
	$user_id = $user_id ? $user_id : get_current_user_id();

	// If user not logged in, bail.
	if ( ! $user_id ) {
		return 0;
	}

	// Get count cache in user meta.
	$count = get_user_meta( $user_id, '_astoundify_favorites_count', true );

	if ( $count ) {
		return $count;
	}

	// Get user favorites data.
	$args = array(
		'user_id' => $user_id,
		'item_per_page' => 1, // minimal.
		'fields' => 'ids',
	);
	$favorite_query = new \Astoundify\Favorites\Favorite_Query( $args );
	$count = intval( $favorite_query->total_items );

	// Update user meta.
	update_user_meta( $user_id, '_astoundify_favorites_count', $count );

	return $count;
}

/**
 * List Drop Down
 *
 * @since 1.0.0
 *
 * @param array $args Modify default arguments.
 */
function astoundify_favorites_list_field( $args = array() ) {
	$defaults = array(
		'selected'          => '',
		// Translators: %s is list.
		'show_option_none'  => sprintf( __( '- Select %s -', 'astoundify-favorites' ), astoundify_favorites_label( 'list' ) ),
		// Translators: %s is list.
		'show_option_new'   => sprintf( __( '+ Create new %s', 'astoundify-favorites' ), astoundify_favorites_label( 'list' ) ),
		// Translators: %s is List.
		'placeholder_new'   => sprintf( __( '%s name', 'astoundify-favorites' ), astoundify_favorites_label( 'List' ) ),
	);

	$args = wp_parse_args( $args, $defaults );

	$favorite_list_query = new \Astoundify\Favorites\Favorite_List_Query();
?>

	<?php if ( empty( $favorite_list_query->lists ) && $args['show_option_new'] ) : ?>

		<div class="astoundify-favorites-list-field">

			<div class="astoundify_favorites_list_new" style="display:block;">
				<input type="text" name="list_new" value="" placeholder="<?php echo esc_attr( $args['placeholder_new'] ); ?>" autocomplete="off"/>
			</div><!-- .astoundify_favorites_list_new -->

		</div><!-- .astoundify-favorites-list-field -->

	<?php else : ?>

		<div class="astoundify-favorites-list-field">

			<select name="list_id" class="astoundify_favorites_list" autocomplete="off">

				<?php if ( $args['show_option_none'] ) : ?>
					<option value=""><?php echo esc_html( $args['show_option_none'] ); ?></option>
				<?php endif; ?>

				<?php foreach ( $favorite_list_query->lists as $list ) : ?>
					<option class="level-0" value="<?php echo esc_attr( $list->get_id() ); ?>" <?php selected( $list->get_id(), $args['selected'] ); ?>><?php echo esc_html( $list->get_name() ); ?></option>
				<?php endforeach; ?>

				<?php if ( $args['show_option_new'] ) : ?>
					<option value="new"><?php echo esc_html( $args['show_option_new'] ); ?></option>
				<?php endif; ?>

			</select><!-- .astoundify_favorites_list -->

			<?php if ( $args['show_option_new'] ) : ?>
				<div class="astoundify_favorites_list_new" style="display:none;">
					<input type="text" name="list_new" value="" placeholder="<?php echo esc_attr( $args['placeholder_new'] ); ?>" autocomplete="off"/>
				</div><!-- .astoundify_favorites_list_new -->
			<?php endif; ?>

		</div><!-- .astoundify-favorites-list-field -->

	<?php endif; ?>

	<?php
}

/**
 * Note Field
 *
 * @since 1.0.0
 *
 * @param array $args Modify default arguments.
 */
function astoundify_favorites_note_field( $args = array() ) {
	$defaults = array(
		'placeholder' => __( 'Add note...', 'astoundify-favorites' ),
		'note'        => '',
		'rows'        => '2',
	);

	$args = wp_parse_args( $args, $defaults );
?>

<textarea class="astoundify-favorites-note-field" autocomplete="off" name="note" rows="<?php esc_attr( $args['rows'] ); ?>" placeholder="<?php echo esc_attr( $args['placeholder'] ); ?>"><?php echo esc_textarea( $args['note'] ); ?></textarea>

<?php
}

/**
 * List Name Field.
 * This is for form-list-edit.php template.
 *
 * @since 1.0.0
 *
 * @param array $args Modify default arguments.
 */
function astoundify_favorites_list_name_field( $args = array() ) {
	$defaults = array(
		// Translators: %s is List.
		'placeholder' => sprintf( __( '%s Name', 'astoundify-favorites' ), astoundify_favorites_label( 'List' ) ),
		'list_name'   => '',
	);

	$args = wp_parse_args( $args, $defaults );
?>

<input type="text" name="list_name" value="<?php echo esc_attr( sanitize_text_field( $args['list_name'] ) ); ?>" placeholder="<?php echo esc_attr( $args['placeholder'] ); ?>" autocomplete="off" required="required"/>

<?php
}


/**
 * Favorite Link
 *
 * @since 1.0.0
 *
 * @param int    $target_id Target Post ID.
 * @param string $before    HTML before link.
 * @param string $after     HTML after link.
 * @return string
 */
function astoundify_favorites_link( $target_id, $before = '', $after = '' ) {
	// Need to have a target ID and not in dashboard.
	if ( ! $target_id || astoundify_favorites_get_option( 'dashboard-page' ) === $target_id ) {
		return '';
	}

	// Check supported post types.
	$post_type = get_post_type( $target_id );

	if ( ! in_array( $post_type, astoundify_favorites_post_types(), true ) ) {
		return '';
	}

	$count = astoundify_favorites_count( $target_id );
	$is_favorited = astoundify_favorites_is_favorited( $target_id, get_current_user_id() );

	if ( 2 > $count ) { // Plural.
		// Translators: %1$s heart SVG, %2$s Number of Favorite, %3$s Favorite singular label.
		$text = sprintf( __( '%1$s %2$s %3$s', 'astoundify-favorites' ), astoundify_favorites_get_svg( 'heart' ), $count, astoundify_favorites_label( 'Favorite' ) );
	} else { // Singular.
		// Translators: %1$s heart SVG, %2$s Number of Favorite, %3$s Favorite plural label.
		$text = sprintf( __( '%1$s %2$s %3$s', 'astoundify-favorites' ), astoundify_favorites_get_svg( 'heart' ), $count, astoundify_favorites_label( 'Favorites' ) );
	}

	// Filter text.
	$text = apply_filters( 'astoundify_favorites_link_text', $text, $target_id, $is_favorited );

	// Get URL for this link.
	if ( ! is_user_logged_in() ) {

		$url = wp_login_url( esc_url( $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] ) );

	} elseif ( $is_favorited ) {

		$favorite = new \Astoundify\Favorites\Favorite( $is_favorited );
		$url = $favorite->get_edit_url();

	} else { // Create new favorite URL.

		$url = add_query_arg( array(
			'af_favorite_id' => 'new',
			'af_data'        => $target_id,
			'_nonce'         => wp_create_nonce( "astoundify_favorites_create_{$target_id}" ),
			'_redirect'      => rawurlencode( esc_url( wp_doing_ajax() ? wp_get_referer() : $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] ) ),
		), astoundify_favorites_dashboard_url() );

	}

	$attr_str = astoundify_favorites_attr( array(
		'href'                => esc_url( $url ),
		'class'               => $is_favorited ? 'astoundify-favorites-link active' : 'astoundify-favorites-link inactive',
		'data-user_id'        => absint( get_current_user_id() ),
		'data-af_favorite_id' => absint( $is_favorited ),
		'data-af_data'        => absint( $target_id ),
		'data-_nonce'         => wp_create_nonce( "astoundify_favorites_create_{$target_id}" ),
	) );

	return "{$before}<a {$attr_str}>{$text}</a>{$after}";
}

/**
 * Get SVG
 *
 * @since 1.0.0
 *
 * @param string $icon Icon name.
 */
function astoundify_favorites_get_svg( $icon ) {
	$file = ASTOUNDIFY_FAVORITES_PATH . "public/images/{$icon}.svg";
	$file = apply_filters( 'astoundify_favorites_svg', $file, $icon );

	if ( file_exists( $file ) ) {
		ob_start();
?>

<span class="astoundify-favorites-icon"><?php require( $file ); ?></span>

<?php
		return ob_get_clean();
	}
}

/**
 * Dashboard URL
 *
 * @since 1.0.0
 *
 * @param string $view The current dashboard view.
 */
function astoundify_favorites_dashboard_url( $view = '' ) {
	$page_id = astoundify_favorites_get_option( 'dashboard-page' );

	// Bail if no dashboard page.
	if ( ! $page_id ) {
		return;
	}

	// Polylang Plugin Support.
	if ( function_exists( 'pll_get_post' ) ) {
		$page_id = pll_get_post( $page_id );
	}

	// Get dashboard URL.
	$page_url = get_permalink( $page_id );

	// Validate views.
	$valid_views = array( 'favorites', 'lists' );
	if ( in_array( $view, $valid_views, true ) ) {
		$page_url = add_query_arg( 'af_view', $view, $page_url );
	}

	return esc_url( $page_url );
}


/**
 * HTML Attribute Helper
 * Convert Array into HTML Attr
 *
 * @since 1.0.0
 *
 * @param array $attr Attributes to build.
 * @return string $attr_str
 */
function astoundify_favorites_attr( $attr ) {
	if ( ! is_array( $attr ) || empty( $attr ) ) {
		return '';
	}

	$attr_str = '';

	foreach ( $attr as $name => $value ) {
		$attr_str .= false !== $value ? sprintf( ' %s="%s"', esc_html( $name ), esc_attr( $value ) ) : esc_html( " {$name}" );
	}

	return $attr_str;
}

/**
 * Display Notices
 *
 * @since 1.0.0
 */
function astoundify_favorites_notices() {
	echo wp_kses_post( \Astoundify\Favorites\Notices::display() );
}

/**
 * Labels
 *
 * @since 1.0.1
 *
 * @param string $for Valid value "favorite", "favorites", "list", "lists".
 * @return string
 */
function astoundify_favorites_label( $for ) {
	if ( 'favorite' === $for ) {
		return esc_attr( astoundify_favorites_get_option( 'favorite-label-singular', __( 'Favorite', 'astoundify-favorites' ) ) );
	} elseif ( 'Favorite' === $for ) {
		return esc_attr( ucfirst( astoundify_favorites_get_option( 'favorite-label-singular', __( 'Favorite', 'astoundify-favorites' ) ) ) );
	} elseif ( 'favorites' === $for ) {
		return esc_attr( astoundify_favorites_get_option( 'favorite-label-plural', __( 'Favorites', 'astoundify-favorites' ) ) );
	} elseif ( 'Favorites' === $for ) {
		return esc_attr( ucfirst( astoundify_favorites_get_option( 'favorite-label-plural', __( 'Favorites', 'astoundify-favorites' ) ) ) );
	} elseif ( 'list' === $for ) {
		return esc_attr( astoundify_favorites_get_option( 'list-label-singular', __( 'List', 'astoundify-favorites' ) ) );
	} elseif ( 'List' === $for ) {
		return esc_attr( ucfirst( astoundify_favorites_get_option( 'list-label-singular', __( 'List', 'astoundify-favorites' ) ) ) );
	} elseif ( 'lists' === $for ) {
		return esc_attr( astoundify_favorites_get_option( 'list-label-plural', __( 'Lists', 'astoundify-favorites' ) ) );
	} elseif ( 'Lists' === $for ) {
		return esc_attr( ucfirst( astoundify_favorites_get_option( 'list-label-plural', __( 'Lists', 'astoundify-favorites' ) ) ) );
	}
}
